<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use App\Models\User;
use App\Service\NotificationService;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Session;
use Laravel\Socialite\Facades\Socialite;

class CustomerAuthController extends Controller
{
    public function showLoginForm()
    {
        return view('frontend.pages.login');
    }

    public function showOtpForm()
    {
        return view('frontend.pages.otp_verify');
    }

    //Google Login
    public function redirectToGoogle()
    {
        return Socialite::driver('google')->stateless()->redirect();
    }

    //Google callback
    public function handleGoogleCallback()
    {
        $user = Socialite::driver('google')->stateless()->user();
        $this->_registerOrLoginUser($user);
        return redirect()->route('customer.dashboard');
    }


    public function sendOtp()
    {
        try {
            $otp = rand(1000, 9999);
            Session::put('otp', $otp);
            if (session()->has('new_register')) {
                $email = session()->get('new_register');
                Mail::send('emails.otp', ['otp' => $otp], function ($message) use ($email) {
                    $message->to($email);
                    $message->subject('OTP Verification');
                });
            }
            return true;
        } catch (\Exception $e) {
            throw $e;
        }
    }

    public function resendOtp()
    {
        try {
            $this->sendOtp();
            return redirect()->back();
        }catch (\Exception $e) {
            throw $e;
        }
    }


    public function sendVerify(Request $request)
    {
        $request->validate([
            'otp_code' => 'required'
        ]);
        $enteredOTP = $request->otp_code;
        $storedOTP = Session::get('otp');
        if (Session::has('new_register')) {
            $register = Session::get('new_register');
        } else {
            $register = Customer::latest()->first();
        }
        if ($enteredOTP == $storedOTP) {
            $customer = Customer::where('email', $register)->first();
            $customer->is_verified = true;
            $customer->save();
            Toastr::success('Your account verified successfully! Please login into your account!', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->route('customer.login');
        } else {
            Toastr::error('Invalid OTP, please try again', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->route('customer.otp.show_form');
        }

    }

    public function loginProcess(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);
        try {
            $customer = Customer::where('email', $request->email)->first();
            if (!empty($customer)) {
                if ($customer->is_verified) {
                    $credentials = $request->only('email', 'password');
                    $res = $this->guard()->attempt($credentials, $request->boolean('remember'));
                    if ($res == false) {
                        Toastr::error('These credentials do not match our records', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
                        return redirect()->back();
                    }
                    return redirect()->route('customer.dashboard');
                } else {
                    Toastr::error('Your account is not verified, Please verify your account!', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
                    session(['new_register' => $request->email]);
                    $this->sendOtp();
                    Toastr::success('OTP has been sent to your email successfully!', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
                    return redirect()->route('customer.otp.show_form');
                }
            } else {
                Toastr::error('These credentials do not match our records', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
                return redirect()->back();
            }
        } catch (\Exception $e) {
            Toastr::error($e->getMessage(), '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->back();
        }
    }


    public function showRegisterForm()
    {
        return view('frontend.pages.register');
    }

    public function registerProcess(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'phone' => 'required',
            'email' => 'required|email|unique:customers',
            'password' => 'required|confirmed|min:6'
        ]);
        try {
            $data = [
                'name' => $request->name,
                'phone' => $request->phone,
                'email' => $request->email,
                'password' => $request->password,
                'type' => 'online'
            ];
           $customer = Customer::create($data);
            NotificationService::send($customer->id,1,'A new customer registered',"$customer->name registered as a customer");
            session(['new_register' => $request->email]);
            $this->sendOtp();
            Toastr::success('OTP has been sent to your email successfully!', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->route('customer.otp.show_form');
        } catch (\Exception $e) {
            Toastr::error($e->getMessage(), '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->back();
        }
    }


    public function logout(Request $request)
    {
        $this->guard()->logout();
        return redirect()->route('customer.login');
    }


    private function guard()
    {
        return Auth::guard('customer');
    }


    protected function _registerOrLoginUser($data)
    {
        $user = User::where('email', $data->email)->first();
        if (!$user) {
            $user = new Customer();
            $user->name = $data->name;
            $user->type = "online";
            $user->email = $data->email;
            $user->is_verified = 1;
            $user->save();
        }
        Auth::guard('customer')->login($user);
    }

}