<?php

namespace App\Http\Controllers;

use App\CPU\Helpers;
use App\Models\Balance;
use App\Models\Batch;
use App\Models\Medicine;
use App\Models\Method;
use App\Models\Purchase;
use App\Models\PurchasePay;
use App\Models\PurchaseReturn;
use App\Models\Returns;
use App\Models\Supplier;
use App\Service\TransactionService;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class NPurchaseController extends Controller
{
    public function index(Request $request)
    {
        $to_date = $request->to_date ?? null;
        $from_date = $request->from_date ?? null;

        $paginate = 10;
        $query = Purchase::with('supplier', 'batch', 'method');
        if (!empty($request->paginate)) {
            $paginate = $request->paginate;
        }
        if (!empty($request->keywords)) {
            $keywords = '%' . $request->keywords . '%';
            $query->where(function ($query) use ($keywords) {
                $query->where('inv_id', 'LIKE', $keywords);
            });
        }
        if (!empty($from_date) && !empty($to_date)){
            $query->whereBetween('created_at', [$from_date, $to_date]);
        }

        $purchases = $query->latest()->paginate($paginate);
        return view('npurchase.index', compact('purchases'));
    }

    public function create()
    {
        $query = Medicine::with('supplier')
            ->select('id', 'name', 'generic_name', 'price', 'image', 'supplier_id', 'strength');
        $search_products = $query->latest()->take(25)->get();
        $purchase_cart = session('purchase_cart', []);
        $invoice_summary = session('invoice_summary', []);
        $suppliers = Supplier::select('id', 'name')->get();

        return view('npurchase.create', compact('search_products',
            'purchase_cart',
            'suppliers',
            'invoice_summary'
        ));
    }


    public function addToCart(Request $request)
    {
        $productId = $request->product_id;
        $product = Medicine::findOrFail($productId);

        $purchase_cart = session('purchase_cart', []);
        $invoice_summary = session('invoice_summary', []);
        $cardData = [
            'id' => $product->id,
            'name' => $product->name,
            'image' => $product->image,
            'price' => $product->price,
            'leaf_id' => $product->leaf_id,
            'vat' => $product->vat,
            'buy_price' => $product->buy_price,
            'quantity' => 1,
            'batch_name' => null,
            'expire_date' => null,
            'discount' => 0,
            'discount_value_type' => 'percent',
            'sub_total' => 0,
            'total' => 0,
        ];
        if (array_key_exists($productId, $purchase_cart)) {
            $purchase_cart[$productId]['quantity'] += 1;
        } else {
            $purchase_cart[$productId] = $cardData;
        }
        session(['purchase_cart' => $purchase_cart]);
        return response()->json([
            'already_has' => 0,
            'added' => 1,
            'view' => view('npurchase.cart_table', compact('purchase_cart','invoice_summary'))->render()
        ]);
    }


    public function removeFromCart(Request $request)
    {
        $productId = $request->product_id;
        $purchase_cart = session('purchase_cart');
        $invoice_summary = session('invoice_summary', []);
        if (array_key_exists($productId, $purchase_cart)) {
            $productId = $request->product_id;
            $purchase_cart = collect(session('purchase_cart')); // convert array to collection
            $purchase_cart->forget($productId); // use forget() method on the collection
            session(['purchase_cart' => $purchase_cart->toArray()]);
            return response()->json([
                'not_exsits' => 0,
                'removed' => 1,
                'view' => view('npurchase.cart_table', compact('purchase_cart','invoice_summary'))->render()
            ]);
        } else {
            return response()->json([
                'not_exsits' => 1,
                'removed' => 0,
                'view' => view('npurchase.cart_table', compact('purchase_cart','invoice_summary'))->render()
            ]);
        }
    }


    public function updateCart(Request $request)
    {
        $cartId = $request->cart_id;
        $field = $request->field;
        $value = $request->value;
        $purchase_cart = session('purchase_cart', []);
        $invoice_summary = session('invoice_summary', []);

        if (array_key_exists($cartId, $purchase_cart)) {
            $purchase_cart[$cartId][$field] = $value;


        }
        session(['purchase_cart' => $purchase_cart]);
        $response = [
            'success' => 1,
            'error' => 0,
            'message' => 'Save changes!',
            'view' => view('npurchase.cart_table', compact('purchase_cart','invoice_summary'))->render()
        ];
        return response()->json($response);
    }

    public function invoiceSummaryChangeHandler(Request $request)
    {
        $invoiceSummary = [
            'invoice_discount_type' => $request->input('invoice_discount_type','percent'),
            'invoice_discount_value' => $request->input('invoice_discount_value',0),
        ];
        session(['invoice_summary' => $invoiceSummary]);

        $purchase_cart = session('purchase_cart', []);
        $invoice_summary = session('invoice_summary', []);

        $response = [
            'success' => 1,
            'error' => 0,
            'message' => 'Save changes!',
            'view' => view('npurchase.cart_table', compact('purchase_cart','invoice_summary'))->render()
        ];
        return response()->json($response);
    }


    public function store(Request $request)
    {
        $this->inputValidate($request);

        try {
            DB::beginTransaction();

            $balance = Method::find($request->input('payment_method_id'));
           if($request->input('paid') > 0 && $balance->balance < $request->input('paid')){
               $availableBalance = priceFormat($balance->balance);
                throw new \Exception("Insufficient balance! your balance on $balance->name : $availableBalance");
           }
           
            $purchase_medicines = session('purchase_cart') ?? [];
            if (empty($purchase_medicines)){
                throw new \Exception('Cart cannot be');
            }
            $data = [
                'date' => $request->purchase_date,
                'total_price' => $request->total,
                'paid_amount' => $request->paid,
                'due_price' => $request->due_amount,
                'change_amount' => $request->change_amount,
                'supplier_id' => $request->supplier_id,
                'qty' => $request->total_quantity,
                'inv_id' => $request->invoice_id,
                'subtotal' => $request->invoice_subtotal_amount,
                'discount' => $request->total_discount,
                'method_id' => $request->payment_method_id,
            ];

            $data['shop_id'] = Auth::user()->shop_id;
            $data['district_id'] = Auth::user()->shop->district_id;
            $data['thana_id'] = Auth::user()->shop->thana_id;
            $purchase = Purchase::create($data);
            if ($purchase) {
                if ($request->due_amount > 0) {
//                    $previus_due = Balance::where('supplier_id', $request->supplier_id)->first();
                    $previus_due = Supplier::findOrFail($request->supplier_id);
                    $previus_due->due += $request->due_amount;
                    $previus_due->save();
                }

                $invpay = new PurchasePay();
                $invpay->shop_id = Auth::user()->shop_id;
                $invpay->purchase_id = $purchase->id;
                $invpay->date = $request->purchase_date;
                $invpay->amount = $request->paid;
                $invpay->supplier_id = $request->supplier_id;
                $invpay->method_id = $request->payment_method_id;
                $invpay->save();

                if ($request->input('paid') > 0){
                    $method = Method::find($request->payment_method_id);
                    $method->balance -= $request->paid;
                    $method->save();
                }

                foreach ($purchase_medicines as $key => $medicine) {
                    $pmd['medicine_id'] = $key;
                    $pmd['purchase_id'] = $purchase->id;
                    $pmd['qty'] = $medicine['quantity'];
                    $pmd['purchase_qty'] = $medicine['quantity'];
                    $pmd['name'] = $medicine['batch_name'];
                    $pmd['price'] = $medicine['price'];
                    $pmd['buy_price'] = $medicine['buy_price'];
                    $pmd['expire'] = $medicine['expire_date'];
                    $pmd['discount'] = $medicine['discount'];
                    $pmd['discount_value_type'] = $medicine['discount_value_type'];
                    Batch::create($pmd);
                }

                TransactionService::purchaseTransaction($purchase->total_price, $purchase->inv_id);

                DB::commit();
                session()->forget('purchase_cart');
                session()->forget('invoice_summary');
                successAlert('Purchase created successfully!');
                return redirect()->route('purchase.index');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            errorAlert($e->getMessage());
            return redirect()->back();
        }
    }


    function show($id)
    {
        $data['invoice'] = Purchase::with('supplier')->where('id', $id)->first();
        $data['purchase_details'] = Batch::where('purchase_id', $id)->get();
        return view('npurchase.view')->with($data);
    }


    public function destroy($id)
    {
        try {
            $purchase = Purchase::find($id);
            if ($purchase->delete()) {
                Batch::where('purchase_id', $id)->delete();
                Toastr::success('Purchase deleted successfully!', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
                return redirect()->route('purchase.index');
            }
        } catch (\Exception $e) {
            Toastr::error($e->getMessage(), '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->back();
        }
    }

    public function returnHistory(Request $request)
    {
        $paginate = 10;
        if (!empty($request->paginate)) {
            $paginate = $request->paginate;
        }
        $return_data = PurchaseReturn::latest('id')->paginate($paginate);
        return view('npurchase.return_list', compact('return_data'));
    }

    public function showReturnForm($id)
    {
        $inv = Purchase::findorFail($id);
        $medicine = Batch::where('purchase_id', $id)->get();
        return view('npurchase.return_form', compact('inv', 'medicine'));
    }

    public function returnProcess(Request $request, $id)
    {
        $request->validate([
            'qty' => 'required',
            'medicine' => 'required',
        ]);
        $inv = Purchase::findorFail($id);
        $batch = Batch::where('medicine_id', $request->medicine)->where('purchase_id', $id)->first();
        if ($request->qty > $batch->qty) {
            Toastr::warning('Invalid quantity applyed! total quantity: ' . $batch->qty . '', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
            return redirect()->back();
        }

        $this->itemReturnMarge($batch, $inv, $request, $id);
        $this->invoiceReturnMarge($batch, $inv, $request, $id);

        Toastr::success('Return Accepted', '', ['progressBar' => true, 'closeButton' => true, 'positionClass' => 'toast-top-right']);
        return redirect()->route('purchase.return');

    }

    protected function itemReturnMarge($item, $invoice, $request, $id)
    {
        $item->qty -= $request->qty;
        $item->save();
        $amt = ($item->buy_price * $request->qty);
        if ($invoice->supplier_id != 0) {
            $customer = Supplier::find($invoice->supplier_id);
            if ($customer->due >= $amt) {
                $customer->due -= $amt;
            }
            $customer->save();
        }
        $return = new PurchaseReturn();
        $return->date = date('Y-m-d');
        $return->purchae_id = $id;
        $return->batch_id = $item->id;
        $return->amount = $amt;
        $return->quantity = $request->qty;
        $return->shop_id = Auth::user()->shop_id;
        return $return->save();
    }

    protected function invoiceReturnMarge($batch, $inv, $request, $id)
    {
        $inv->qty -= $request->qty;
        $inv->total_price -= $batch->buy_price * $request->qty;
        $inv->subtotal -= $batch->buy_price * $request->qty;
        return $inv->save();
    }


    public function returnInvoice($id)
    {
        $return = PurchaseReturn::findOrFail($id);
        $invoice = Purchase::find($return->purchae_id);
        $purchase_details = Batch::where('purchase_id', $invoice->id)->get();
        return view('npurchase.return_invoice', compact('return', 'invoice','purchase_details'));

    }


    public function getMedicines(Request $request)
    {
        $query = Medicine::with('supplier')->select('id', 'name', 'generic_name', 'price', 'image', 'supplier_id', 'strength');
        if (!empty($request->keywords)) {
            $keywords = '%' . $request->keywords . '%';
            $query->where(function ($query) use ($keywords) {
                $query->where('name', 'LIKE', $keywords)
                    ->orWhere('generic_name', 'LIKE', $keywords);
            });
        }
        $search_products = $query->get();
        return response([
            'results' => view('npurchase.search_result', compact('search_products'))->render(),
        ]);
    }


    public function calculateCharge($value, $amount, $type)
    {
        if ($value && $type == 'percent') {
            return ($value / 100) * $amount;
        }
        if ($value && $type == 'fixed') {
            return $value;
        }
        return 0;
    }


    protected function inputValidate($request)
    {
        $rules = [
            'purchase_date' => 'required',
            'invoice_id' => 'required',
            'supplier_id' => 'required',
            'invoice_subtotal_amount' => 'required',
            'total' => 'required',
        ];
        if ($request->input('paid') > 1) {
            $rules['payment_method_id'] = 'required';
        }

        $messages = [
            'purchase_date.required' => 'The purchase date is required.',
            'invoice_id.required' => 'The invoice ID is required.',
            'supplier_id.required' => 'The Manufacture is required.',
            'invoice_subtotal_amount.required' => 'The invoice subtotal amount is required.',
            'total.required' => 'The total amount is required.',
            'payment_method_id.required' => 'The payment method is required when the paid amount exceeds 1.',
        ];

        return $request->validate($rules, $messages);
    }

}